var __defProp = Object.defineProperty;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __publicField = (obj, key, value) => __defNormalProp(obj, typeof key !== "symbol" ? key + "" : key, value);
import { P as PUBLIC_RPC_ENDPOINTS, s as sleep, r as retry, D as DEXSCREENER_BASE, A as ANALYSIS_THRESHOLDS, a as PUMPFUN_PROGRAM_ID } from "./chunks/utils.C3mUlwWA.js";
class SolanaRpcClient {
  constructor(customEndpoint) {
    __publicField(this, "endpoints");
    __publicField(this, "currentIndex", 0);
    __publicField(this, "requestCount", 0);
    __publicField(this, "lastRequestTime", 0);
    this.endpoints = customEndpoint ? [customEndpoint, ...PUBLIC_RPC_ENDPOINTS] : [...PUBLIC_RPC_ENDPOINTS];
  }
  async request(method, params) {
    const now = Date.now();
    if (now - this.lastRequestTime < 200) {
      await sleep(200 - (now - this.lastRequestTime));
    }
    this.lastRequestTime = Date.now();
    this.requestCount++;
    let lastError = null;
    const startIndex = this.currentIndex;
    for (let attempt = 0; attempt < this.endpoints.length; attempt++) {
      const endpointIndex = (startIndex + attempt) % this.endpoints.length;
      const endpoint = this.endpoints[endpointIndex];
      try {
        const response = await fetch(endpoint, {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          body: JSON.stringify({
            jsonrpc: "2.0",
            id: this.requestCount,
            method,
            params
          })
        });
        if (!response.ok) {
          console.log(`[RPC] ${endpoint} 返回 ${response.status}，尝试下一个端点`);
          lastError = new Error(`RPC 请求失败: ${response.status}`);
          continue;
        }
        const data = await response.json();
        if (data.error) {
          console.log(`[RPC] ${endpoint} 返回错误: ${data.error.message}`);
          lastError = new Error(`RPC 错误: ${data.error.message}`);
          continue;
        }
        this.currentIndex = endpointIndex;
        return data.result;
      } catch (err) {
        console.log(`[RPC] ${endpoint} 请求失败:`, err);
        lastError = err instanceof Error ? err : new Error(String(err));
        continue;
      }
    }
    throw lastError || new Error("所有 RPC 端点都不可用");
  }
  // 获取地址的交易签名列表
  async getSignaturesForAddress(address, options = {}) {
    return retry(
      () => this.request("getSignaturesForAddress", [
        address,
        {
          limit: options.limit || 100,
          ...options.before && { before: options.before },
          ...options.until && { until: options.until }
        }
      ]),
      3,
      1e3
    );
  }
  // 获取交易详情
  async getTransaction(signature) {
    try {
      return await retry(
        () => this.request("getTransaction", [
          signature,
          {
            encoding: "json",
            maxSupportedTransactionVersion: 0
          }
        ]),
        2,
        500
      );
    } catch {
      return null;
    }
  }
  // 批量获取交易（带速率限制）
  async getTransactions(signatures, onProgress) {
    const results = [];
    const batchSize = 5;
    for (let i = 0; i < signatures.length; i += batchSize) {
      const batch = signatures.slice(i, i + batchSize);
      const batchResults = await Promise.all(
        batch.map((sig) => this.getTransaction(sig))
      );
      results.push(...batchResults);
      if (onProgress) {
        onProgress(Math.min(i + batchSize, signatures.length), signatures.length);
      }
      if (i + batchSize < signatures.length) {
        await sleep(500);
      }
    }
    return results;
  }
  // 获取账户信息
  async getAccountInfo(address) {
    return retry(
      () => this.request("getAccountInfo", [
        address,
        { encoding: "jsonParsed" }
      ]),
      2,
      500
    );
  }
  // 获取代币账户
  async getTokenAccountsByOwner(owner, mint) {
    const filter = mint ? { mint } : { programId: "TokenkegQfeZyiNwAJbNbGKPFXCWuBvf9Ss623VQ5DA" };
    const result = await retry(
      () => this.request("getTokenAccountsByOwner", [
        owner,
        filter,
        { encoding: "jsonParsed" }
      ]),
      2,
      500
    );
    return (result == null ? void 0 : result.value) || [];
  }
  // 获取 SOL 余额
  async getBalance(address) {
    const result = await retry(
      () => this.request("getBalance", [address]),
      2,
      500
    );
    return ((result == null ? void 0 : result.value) || 0) / 1e9;
  }
  // 获取最近区块哈希（用于判断网络状态）
  async getLatestBlockhash() {
    return retry(
      () => this.request(
        "getLatestBlockhash",
        []
      ).then((r) => r.value),
      2,
      500
    );
  }
  // 检查 RPC 健康状态
  async checkHealth() {
    try {
      const result = await this.request("getHealth", []);
      return result === "ok";
    } catch {
      return false;
    }
  }
}
let rpcClient = null;
function getRpcClient(customEndpoint) {
  if (!rpcClient || customEndpoint) {
    rpcClient = new SolanaRpcClient(customEndpoint);
  }
  return rpcClient;
}
class DexScreenerClient {
  constructor() {
    __publicField(this, "lastRequestTime", 0);
  }
  async request(url) {
    const now = Date.now();
    if (now - this.lastRequestTime < 500) {
      await sleep(500 - (now - this.lastRequestTime));
    }
    this.lastRequestTime = Date.now();
    const response = await fetch(url);
    if (!response.ok) {
      throw new Error(`DexScreener 请求失败: ${response.status}`);
    }
    return response.json();
  }
  // 根据代币地址获取交易对信息
  async getTokenPairs(tokenAddress) {
    try {
      const data = await retry(
        () => this.request(
          `${DEXSCREENER_BASE}/latest/dex/tokens/${tokenAddress}`
        ),
        2,
        1e3
      );
      return (data.pairs || []).filter((pair) => pair.chainId === "solana");
    } catch {
      return [];
    }
  }
  // 分析单个代币的 DEX 表现
  async analyzeToken(mint) {
    const allPairs = await this.getTokenPairs(mint);
    const realDexPairs = allPairs.filter(
      (p) => p.dexId !== "pumpfun" && p.dexId !== "pump" && p.dexId !== "moonshot"
    );
    const isOnRealDex = realDexPairs.length > 0;
    const pairsToUse = isOnRealDex ? realDexPairs : allPairs;
    if (pairsToUse.length === 0) {
      return {
        mint,
        pairs: [],
        isListed: false,
        maxFdv: 0,
        currentLiquidity: 0,
        volume24h: 0
      };
    }
    const maxFdv = Math.max(
      ...pairsToUse.map((p) => p.fdv || p.marketCap || 0)
    );
    const currentLiquidity = pairsToUse.reduce(
      (sum, p) => {
        var _a;
        return sum + (((_a = p.liquidity) == null ? void 0 : _a.usd) || 0);
      },
      0
    );
    const volume24h = pairsToUse.reduce(
      (sum, p) => {
        var _a;
        return sum + (((_a = p.volume) == null ? void 0 : _a.h24) || 0);
      },
      0
    );
    return {
      mint,
      pairs: allPairs,
      // 返回所有 pairs 供显示
      isListed: isOnRealDex,
      // 只有真正上外盘才算 listed
      maxFdv,
      currentLiquidity,
      volume24h
    };
  }
  // 批量分析多个代币
  async analyzeTokens(mints, onProgress) {
    const tokens = [];
    let listedCount = 0;
    let bestPerformer;
    for (let i = 0; i < mints.length; i++) {
      const tokenInfo = await this.analyzeToken(mints[i]);
      tokens.push(tokenInfo);
      if (tokenInfo.isListed) {
        listedCount++;
        if (!bestPerformer || tokenInfo.maxFdv > bestPerformer.maxFdv) {
          bestPerformer = tokenInfo;
        }
      }
      if (onProgress) {
        onProgress(i + 1, mints.length);
      }
      if (i < mints.length - 1) {
        await sleep(300);
      }
    }
    return {
      listedCount,
      successRate: mints.length > 0 ? listedCount / mints.length : 0,
      totalTokens: mints.length,
      tokens,
      bestPerformer
    };
  }
  // 搜索代币
  async searchTokens(query) {
    try {
      const data = await retry(
        () => this.request(
          `${DEXSCREENER_BASE}/latest/dex/search?q=${encodeURIComponent(query)}`
        ),
        2,
        1e3
      );
      return (data.pairs || []).filter((pair) => pair.chainId === "solana");
    } catch {
      return [];
    }
  }
  // 根据交易对地址获取信息
  async getPairInfo(pairAddress) {
    try {
      const data = await retry(
        () => this.request(
          `${DEXSCREENER_BASE}/latest/dex/pairs/solana/${pairAddress}`
        ),
        2,
        1e3
      );
      return data.pair;
    } catch {
      return null;
    }
  }
}
let dexClient = null;
function getDexClient() {
  if (!dexClient) {
    dexClient = new DexScreenerClient();
  }
  return dexClient;
}
class CacheManager {
  constructor() {
    __publicField(this, "memoryCache", /* @__PURE__ */ new Map());
  }
  // 获取缓存
  async get(key) {
    const memEntry = this.memoryCache.get(key);
    if (memEntry) {
      if (Date.now() - memEntry.timestamp < memEntry.ttl) {
        return memEntry.data;
      } else {
        this.memoryCache.delete(key);
      }
    }
    try {
      const result = await chrome.storage.local.get(key);
      const entry = result[key];
      if (!entry) return null;
      if (Date.now() - entry.timestamp > entry.ttl) {
        await this.delete(key);
        return null;
      }
      this.memoryCache.set(key, entry);
      return entry.data;
    } catch (error) {
      console.error("读取缓存失败:", error);
      return null;
    }
  }
  // 设置缓存
  async set(key, data, ttl = ANALYSIS_THRESHOLDS.CACHE_TTL) {
    const entry = {
      data,
      timestamp: Date.now(),
      ttl
    };
    this.memoryCache.set(key, entry);
    try {
      await chrome.storage.local.set({ [key]: entry });
    } catch (error) {
      console.error("写入缓存失败:", error);
    }
  }
  // 删除缓存
  async delete(key) {
    this.memoryCache.delete(key);
    try {
      await chrome.storage.local.remove(key);
    } catch (error) {
      console.error("删除缓存失败:", error);
    }
  }
  // 清除所有缓存
  async clear() {
    this.memoryCache.clear();
    try {
      await chrome.storage.local.clear();
    } catch (error) {
      console.error("清除缓存失败:", error);
    }
  }
  // 获取缓存大小
  async getSize() {
    try {
      const bytesInUse = await chrome.storage.local.getBytesInUse();
      return bytesInUse;
    } catch {
      return 0;
    }
  }
}
class DevAnalysisCache {
  constructor() {
    __publicField(this, "cache");
    __publicField(this, "PREFIX", "dev_analysis_");
    this.cache = new CacheManager();
  }
  async get(devAddress) {
    return this.cache.get(this.PREFIX + devAddress);
  }
  async set(result) {
    await this.cache.set(this.PREFIX + result.devAddress, result);
  }
  async delete(devAddress) {
    await this.cache.delete(this.PREFIX + devAddress);
  }
  async clear() {
    const all = await chrome.storage.local.get();
    const keysToDelete = Object.keys(all).filter(
      (key) => key.startsWith(this.PREFIX)
    );
    if (keysToDelete.length > 0) {
      await chrome.storage.local.remove(keysToDelete);
    }
  }
}
let analysisCache = null;
function getAnalysisCache() {
  if (!analysisCache) {
    analysisCache = new DevAnalysisCache();
  }
  return analysisCache;
}
const KNOWN_CEX_WALLETS = {
  // Binance
  "5tzFkiKscXHK5ZXCGbXZxdw7gTjjD1mBwuoFbhUvuAi9": "Binance",
  "9WzDXwBbmkg8ZTbNMqUxvQRAyrZzDsGYdLVL9zYtAWWM": "Binance",
  "2ojv9BAiHUrvsm9gxDe7fJSzbNZSJcxZvf8dqmWGHG8S": "Binance",
  // OKX
  "5VCwKtCXgCJ6kit5FybXjvriW3xELsFDhYrPSqtJNmcD": "OKX",
  // Bybit
  "AC5RDfQFmDS1deWZos921JfqscXdByf8BKHs5ACWjtW2": "Bybit",
  // KuCoin
  "BmFdpraQhkiDQE6SnfG5omcA1VwzqfXrwtNYBwWTymy6": "KuCoin",
  // Gate.io
  "u6PJ8DtQuPFnfmwHbGFULQ4u4EgjDiyYKjVEsynXq2w": "Gate.io",
  // MEXC
  "ASTyfSima4LLAdDgoFGkgqoKowG1LZFDr9fAQrg7iaJZ": "MEXC"
};
const DEAD_TOKEN_LIQUIDITY_THRESHOLD = 500;
const QUICK_RUG_THRESHOLD = 24 * 60 * 60 * 1e3;
let currentDevAddress = null;
let currentAnalysis = null;
chrome.action.onClicked.addListener(async (tab) => {
  if (tab.id) {
    await chrome.sidePanel.open({ tabId: tab.id });
  }
});
chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true });
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  handleMessage(message).then(sendResponse);
  return true;
});
const HISTORY_STORAGE_KEY = "dev_analysis_history";
const MAX_HISTORY_COUNT = 50;
async function handleMessage(message, _sender) {
  console.log("[Background] 收到消息:", message.type);
  switch (message.type) {
    case "ANALYZE_DEV":
      return handleAnalyzeDev(message);
    case "GET_CURRENT_DEV":
      return { devAddress: currentDevAddress, analysis: currentAnalysis };
    case "DEV_INFO_DETECTED":
      const payload = message.payload;
      console.log("[Background] 检测到 Dev:", payload.devAddress);
      currentDevAddress = payload.devAddress;
      broadcastToSidepanel({ type: "DEV_INFO_DETECTED", payload });
      return { success: true };
    case "CLEAR_CACHE":
      console.log("[Background] 清除缓存");
      const cache = getAnalysisCache();
      await cache.clear();
      currentAnalysis = null;
      return { success: true };
    case "GET_HISTORY":
      return getHistory();
    case "SAVE_HISTORY":
      return { success: true };
    default:
      return { error: "未知消息类型" };
  }
}
async function getHistory() {
  try {
    const result = await chrome.storage.local.get(HISTORY_STORAGE_KEY);
    return result[HISTORY_STORAGE_KEY] || [];
  } catch {
    return [];
  }
}
async function saveToHistory(result) {
  try {
    const history = await getHistory();
    const entry = {
      devAddress: result.devAddress,
      score: result.score.overall,
      grade: result.score.grade,
      label: result.score.label,
      tokenCount: result.cluster.tokensCreated.length,
      analyzedAt: result.analyzedAt
    };
    const filteredHistory = history.filter((h) => h.devAddress !== entry.devAddress);
    filteredHistory.unshift(entry);
    const trimmedHistory = filteredHistory.slice(0, MAX_HISTORY_COUNT);
    await chrome.storage.local.set({ [HISTORY_STORAGE_KEY]: trimmedHistory });
  } catch (e) {
    console.error("[Background] 保存历史记录失败:", e);
  }
}
async function handleAnalyzeDev(message) {
  var _a, _b;
  const { devAddress, tokenMint, forceRefresh } = message.payload;
  console.log("[Background] 开始分析 Dev:", devAddress, forceRefresh ? "(强制刷新)" : "");
  currentDevAddress = devAddress;
  broadcastToSidepanel({
    type: "ANALYSIS_PROGRESS",
    payload: { stage: "开始", progress: 0, message: "正在初始化..." }
  });
  try {
    const cache = getAnalysisCache();
    if (!forceRefresh) {
      const cached = await cache.get(devAddress);
      if (cached) {
        console.log("[Background] 使用缓存");
        currentAnalysis = cached;
        return cached;
      }
    } else {
      console.log("[Background] 强制刷新，删除旧缓存");
      await cache.delete(devAddress);
    }
    const rpc = getRpcClient();
    const dex = getDexClient();
    broadcastToSidepanel({
      type: "ANALYSIS_PROGRESS",
      payload: { stage: "查询链上数据", progress: 5, message: "正在获取交易历史..." }
    });
    const tokensCreated = [];
    const fundingSources = [];
    let transactions = [];
    try {
      const signatures = await rpc.getSignaturesForAddress(devAddress, { limit: 200 });
      console.log("[Background] 获取到交易签名:", signatures.length);
      let analyzed = 0;
      for (const sig of signatures.slice(0, 100)) {
        try {
          const tx = await rpc.getTransaction(sig.signature);
          if (!tx) continue;
          transactions.push(tx);
          if (isPumpfunCreateTx(tx)) {
            const mintAddress = extractMintFromTx(tx);
            if (mintAddress && !tokensCreated.find((t) => t.mint === mintAddress)) {
              tokensCreated.push({
                mint: mintAddress,
                name: "未知",
                symbol: "???",
                createdAt: tx.blockTime ? tx.blockTime * 1e3 : Date.now(),
                creator: devAddress
              });
            }
          }
          const fundingSource = analyzeFundingFromTx(tx, devAddress);
          if (fundingSource) {
            fundingSources.push(fundingSource);
          }
        } catch (e) {
        }
        analyzed++;
        if (analyzed % 20 === 0) {
          broadcastToSidepanel({
            type: "ANALYSIS_PROGRESS",
            payload: {
              stage: "分析交易历史",
              progress: 5 + Math.round(analyzed / 100 * 25),
              message: `已分析 ${analyzed}/100 笔交易...`
            }
          });
        }
      }
    } catch (rpcError) {
      console.error("[Background] RPC 查询失败:", rpcError);
    }
    if (tokensCreated.length === 0 && tokenMint) {
      tokensCreated.push({
        mint: tokenMint,
        name: "当前代币",
        symbol: "???",
        createdAt: Date.now(),
        creator: devAddress
      });
    }
    console.log("[Background] 找到代币数量:", tokensCreated.length);
    console.log("[Background] 找到资金来源:", fundingSources.length);
    broadcastToSidepanel({
      type: "ANALYSIS_PROGRESS",
      payload: { stage: "分析持仓", progress: 35, message: "正在获取当前持仓..." }
    });
    const holdingsAnalysis = await analyzeHoldings(rpc, dex, devAddress);
    broadcastToSidepanel({
      type: "ANALYSIS_PROGRESS",
      payload: { stage: "分析 DEX 数据", progress: 45, message: "正在查询 DEX 上架情况..." }
    });
    const dexTokens = [];
    const rugTokens = [];
    let listedCount = 0;
    let aliveCount = 0;
    let deadCount = 0;
    const survivalTimes = [];
    const rugSpeeds = [];
    let estimatedRugProfits = 0;
    let socialInfo;
    for (let i = 0; i < Math.min(tokensCreated.length, 30); i++) {
      const token = tokensCreated[i];
      try {
        const dexInfo = await dex.analyzeToken(token.mint);
        dexTokens.push(dexInfo);
        if (dexInfo.pairs.length > 0) {
          token.name = ((_a = dexInfo.pairs[0].baseToken) == null ? void 0 : _a.name) || token.name;
          token.symbol = ((_b = dexInfo.pairs[0].baseToken) == null ? void 0 : _b.symbol) || token.symbol;
          if (!socialInfo && dexInfo.pairs[0]) {
            socialInfo = extractSocialInfo(dexInfo.pairs[0]);
          }
        }
        if (dexInfo.isListed) {
          listedCount++;
        }
        const isAlive = dexInfo.currentLiquidity >= DEAD_TOKEN_LIQUIDITY_THRESHOLD;
        if (isAlive) {
          aliveCount++;
          survivalTimes.push(Date.now() - token.createdAt);
        } else {
          deadCount++;
          const survivalTime = estimateTokenSurvivalTime(dexInfo, token.createdAt);
          if (survivalTime > 0) {
            survivalTimes.push(survivalTime);
          }
        }
        if (dexInfo.pairs.length > 0 && dexInfo.currentLiquidity < 100) {
          const rugSpeed = Date.now() - token.createdAt;
          const isQuickRug = rugSpeed < QUICK_RUG_THRESHOLD;
          rugTokens.push({
            mint: token.mint,
            name: token.name,
            rugType: isQuickRug ? "quick_dump" : "abandoned",
            evidence: generateRugEvidence(dexInfo, rugSpeed),
            createdAt: token.createdAt,
            ruggedAt: token.createdAt + rugSpeed,
            survivalTime: rugSpeed
          });
          rugSpeeds.push(rugSpeed);
          if (dexInfo.maxFdv > 0) {
            estimatedRugProfits += dexInfo.maxFdv * 0.05;
          }
        }
        broadcastToSidepanel({
          type: "ANALYSIS_PROGRESS",
          payload: {
            stage: "分析 DEX 数据",
            progress: 45 + Math.round((i + 1) / Math.min(tokensCreated.length, 30) * 45),
            message: `已分析 ${i + 1}/${Math.min(tokensCreated.length, 30)} 个代币...`
          }
        });
      } catch (e) {
        console.error("[Background] 分析代币失败:", token.mint, e);
      }
    }
    const fundingAnalysis = {
      sources: fundingSources.sort((a, b) => b.timestamp - a.timestamp).slice(0, 10),
      totalFunding: fundingSources.reduce((sum, s) => sum + s.amount, 0),
      firstFundingTime: fundingSources.length > 0 ? Math.min(...fundingSources.map((s) => s.timestamp)) : 0,
      primarySource: fundingSources.sort((a, b) => b.amount - a.amount)[0]
    };
    const cluster = {
      primaryDev: devAddress,
      fundingSources: fundingSources.map((s) => s.address),
      bundleWallets: [],
      tokensCreated
    };
    const avgRugSpeed = rugSpeeds.length > 0 ? rugSpeeds.reduce((a, b) => a + b, 0) / rugSpeeds.length : void 0;
    const fastestRug = rugSpeeds.length > 0 ? Math.min(...rugSpeeds) : void 0;
    const slowestRug = rugSpeeds.length > 0 ? Math.max(...rugSpeeds) : void 0;
    const rugAnalysis = {
      suspectedRugCount: rugTokens.length,
      rugRatio: tokensCreated.length > 0 ? rugTokens.length / tokensCreated.length : 0,
      rugTokens,
      avgRugSpeed,
      fastestRug,
      slowestRug,
      estimatedRugProfits
    };
    const avgSurvivalTime = survivalTimes.length > 0 ? survivalTimes.reduce((a, b) => a + b, 0) / survivalTimes.length : void 0;
    const dexAnalysis = {
      listedCount,
      successRate: tokensCreated.length > 0 ? listedCount / tokensCreated.length : 0,
      totalTokens: tokensCreated.length,
      tokens: dexTokens,
      avgSurvivalTime,
      aliveCount,
      deadCount
    };
    if (dexTokens.length > 0) {
      dexAnalysis.bestPerformer = dexTokens.reduce(
        (best, curr) => curr.maxFdv > ((best == null ? void 0 : best.maxFdv) || 0) ? curr : best,
        dexTokens[0]
      );
    }
    const score = calculateScore(cluster, rugAnalysis, dexAnalysis);
    const result = {
      devAddress,
      cluster,
      rugAnalysis,
      dexAnalysis,
      score,
      analyzedAt: Date.now(),
      fundingAnalysis,
      holdingsAnalysis,
      socialInfo
    };
    console.log("[Background] 分析完成:", {
      tokensCreated: tokensCreated.length,
      listedCount,
      rugCount: rugTokens.length,
      score: score.overall,
      fundingSources: fundingSources.length,
      holdings: holdingsAnalysis.tokenHoldings.length
    });
    await cache.set(result);
    currentAnalysis = result;
    await saveToHistory(result);
    broadcastToSidepanel({
      type: "ANALYSIS_PROGRESS",
      payload: { stage: "完成", progress: 100, message: "分析完成" }
    });
    return result;
  } catch (error) {
    console.error("[Background] 分析失败:", error);
    return createErrorResult(devAddress, error);
  }
}
function analyzeFundingFromTx(tx, devAddress) {
  var _a, _b, _c, _d;
  try {
    const meta = tx.meta;
    if (!meta || meta.err) return null;
    const accountKeys = ((_b = (_a = tx.transaction) == null ? void 0 : _a.message) == null ? void 0 : _b.accountKeys) || [];
    const preBalances = meta.preBalances || [];
    const postBalances = meta.postBalances || [];
    let devIndex = -1;
    for (let i = 0; i < accountKeys.length; i++) {
      const key = typeof accountKeys[i] === "string" ? accountKeys[i] : (_c = accountKeys[i]) == null ? void 0 : _c.pubkey;
      if (key === devAddress) {
        devIndex = i;
        break;
      }
    }
    if (devIndex === -1) return null;
    const balanceChange = (postBalances[devIndex] - preBalances[devIndex]) / 1e9;
    if (balanceChange < 0.01) return null;
    let sourceAddress = null;
    let maxDecrease = 0;
    for (let i = 0; i < accountKeys.length; i++) {
      if (i === devIndex) continue;
      const decrease = (preBalances[i] - postBalances[i]) / 1e9;
      if (decrease > maxDecrease && decrease > 0.01) {
        maxDecrease = decrease;
        sourceAddress = typeof accountKeys[i] === "string" ? accountKeys[i] : (_d = accountKeys[i]) == null ? void 0 : _d.pubkey;
      }
    }
    if (!sourceAddress) return null;
    let type = "wallet";
    let label;
    if (KNOWN_CEX_WALLETS[sourceAddress]) {
      type = "cex";
      label = KNOWN_CEX_WALLETS[sourceAddress];
    }
    return {
      address: sourceAddress,
      amount: balanceChange,
      timestamp: tx.blockTime ? tx.blockTime * 1e3 : Date.now(),
      type,
      label
    };
  } catch {
    return null;
  }
}
async function analyzeHoldings(rpc, dex, devAddress) {
  var _a, _b, _c, _d, _e, _f, _g;
  const holdings = [];
  let solBalance = 0;
  let totalValueUsd = 0;
  try {
    solBalance = await rpc.getBalance(devAddress);
    const solPrice = 200;
    totalValueUsd += solBalance * solPrice;
    const tokenAccounts = await rpc.getTokenAccountsByOwner(devAddress);
    for (const account of tokenAccounts.slice(0, 20)) {
      try {
        const info = (_c = (_b = (_a = account == null ? void 0 : account.account) == null ? void 0 : _a.data) == null ? void 0 : _b.parsed) == null ? void 0 : _c.info;
        if (!info) continue;
        const mint = info.mint;
        const balance = parseFloat(((_d = info.tokenAmount) == null ? void 0 : _d.uiAmount) || "0");
        const decimals = ((_e = info.tokenAmount) == null ? void 0 : _e.decimals) || 0;
        if (balance <= 0) continue;
        let valueUsd = 0;
        let symbol = "???";
        let name = "未知";
        try {
          const dexInfo = await dex.analyzeToken(mint);
          if (dexInfo.pairs.length > 0) {
            const pair = dexInfo.pairs[0];
            const price = parseFloat(pair.priceUsd || "0");
            valueUsd = balance * price;
            symbol = ((_f = pair.baseToken) == null ? void 0 : _f.symbol) || symbol;
            name = ((_g = pair.baseToken) == null ? void 0 : _g.name) || name;
          }
        } catch {
        }
        holdings.push({
          mint,
          symbol,
          name,
          balance,
          valueUsd,
          decimals
        });
        totalValueUsd += valueUsd;
      } catch {
      }
    }
  } catch (e) {
    console.error("[Background] 获取持仓失败:", e);
  }
  holdings.sort((a, b) => b.valueUsd - a.valueUsd);
  return {
    solBalance,
    tokenHoldings: holdings.slice(0, 10),
    totalValueUsd
  };
}
function extractSocialInfo(pair) {
  var _a;
  const info = pair.info;
  if (!info) return void 0;
  const social = {};
  if (((_a = info.websites) == null ? void 0 : _a.length) > 0) {
    social.website = info.websites[0].url;
  }
  if (info.socials) {
    for (const s of info.socials) {
      if (s.type === "twitter") social.twitter = s.url;
      if (s.type === "telegram") social.telegram = s.url;
      if (s.type === "discord") social.discord = s.url;
    }
  }
  return Object.keys(social).length > 0 ? social : void 0;
}
function estimateTokenSurvivalTime(dexInfo, createdAt) {
  if (dexInfo.pairs.length > 0 && dexInfo.pairs[0].pairCreatedAt) {
    return dexInfo.pairs[0].pairCreatedAt - createdAt;
  }
  return 7 * 24 * 60 * 60 * 1e3;
}
function generateRugEvidence(dexInfo, rugSpeed) {
  const evidence = [];
  if (dexInfo.currentLiquidity < 100) {
    evidence.push(`当前流动性仅 $${dexInfo.currentLiquidity.toFixed(2)}`);
  }
  if (rugSpeed < 60 * 60 * 1e3) {
    evidence.push(`发币后 ${Math.round(rugSpeed / 6e4)} 分钟内清仓`);
  } else if (rugSpeed < 24 * 60 * 60 * 1e3) {
    evidence.push(`发币后 ${Math.round(rugSpeed / 36e5)} 小时内清仓`);
  } else {
    evidence.push(`发币后 ${Math.round(rugSpeed / 864e5)} 天后弃置`);
  }
  if (dexInfo.maxFdv > 1e4 && dexInfo.currentLiquidity < 100) {
    evidence.push(`最高市值 $${(dexInfo.maxFdv / 1e3).toFixed(1)}K，现已归零`);
  }
  return evidence;
}
const PUMPFUN_CREATE_DISCRIMINATOR = [24, 30, 200, 40, 5, 28, 7, 119];
function isPumpfunCreateTx(tx) {
  var _a, _b, _c, _d, _e, _f, _g;
  try {
    const accountKeys = ((_b = (_a = tx.transaction) == null ? void 0 : _a.message) == null ? void 0 : _b.accountKeys) || [];
    const instructions = ((_d = (_c = tx.transaction) == null ? void 0 : _c.message) == null ? void 0 : _d.instructions) || [];
    let pumpfunIndex = -1;
    for (let i = 0; i < accountKeys.length; i++) {
      const key = typeof accountKeys[i] === "string" ? accountKeys[i] : (_e = accountKeys[i]) == null ? void 0 : _e.pubkey;
      if (key === PUMPFUN_PROGRAM_ID) {
        pumpfunIndex = i;
        break;
      }
    }
    if (pumpfunIndex === -1) return false;
    for (const ix of instructions) {
      if (ix.programIdIndex === pumpfunIndex && ix.data) {
        const decoded = decodeBase58(ix.data);
        if (decoded && decoded.length >= 8) {
          let isCreate = true;
          for (let i = 0; i < 8; i++) {
            if (decoded[i] !== PUMPFUN_CREATE_DISCRIMINATOR[i]) {
              isCreate = false;
              break;
            }
          }
          if (isCreate) return true;
        }
      }
    }
    const preTokenBalances = ((_f = tx.meta) == null ? void 0 : _f.preTokenBalances) || [];
    const postTokenBalances = ((_g = tx.meta) == null ? void 0 : _g.postTokenBalances) || [];
    const preMints = new Set(preTokenBalances.map((b) => b.mint));
    for (const postBal of postTokenBalances) {
      if (postBal.mint && !preMints.has(postBal.mint)) {
        const hasPumpfun = accountKeys.some((key) => {
          const pubkey = typeof key === "string" ? key : key == null ? void 0 : key.pubkey;
          return pubkey === PUMPFUN_PROGRAM_ID;
        });
        if (hasPumpfun) return true;
      }
    }
    return false;
  } catch {
    return false;
  }
}
function decodeBase58(str) {
  const ALPHABET = "123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz";
  const ALPHABET_MAP = {};
  for (let i = 0; i < ALPHABET.length; i++) {
    ALPHABET_MAP[ALPHABET[i]] = i;
  }
  try {
    const bytes = [0];
    for (const char of str) {
      const value = ALPHABET_MAP[char];
      if (value === void 0) return null;
      let carry = value;
      for (let j = 0; j < bytes.length; j++) {
        carry += bytes[j] * 58;
        bytes[j] = carry & 255;
        carry >>= 8;
      }
      while (carry > 0) {
        bytes.push(carry & 255);
        carry >>= 8;
      }
    }
    for (const char of str) {
      if (char !== "1") break;
      bytes.push(0);
    }
    return bytes.reverse();
  } catch {
    return null;
  }
}
function extractMintFromTx(tx) {
  var _a, _b, _c, _d, _e, _f;
  try {
    const preTokenBalances = ((_a = tx.meta) == null ? void 0 : _a.preTokenBalances) || [];
    const postTokenBalances = ((_b = tx.meta) == null ? void 0 : _b.postTokenBalances) || [];
    const preMints = new Set(preTokenBalances.map((b) => b.mint));
    for (const postBal of postTokenBalances) {
      if (postBal.mint && !preMints.has(postBal.mint)) {
        return postBal.mint;
      }
    }
    for (const postBal of postTokenBalances) {
      if (postBal.mint && ((_c = postBal.uiTokenAmount) == null ? void 0 : _c.uiAmount) && postBal.uiTokenAmount.uiAmount > 0) {
        return postBal.mint;
      }
    }
    const accountKeys = ((_e = (_d = tx.transaction) == null ? void 0 : _d.message) == null ? void 0 : _e.accountKeys) || [];
    const excludedAddresses = /* @__PURE__ */ new Set([
      PUMPFUN_PROGRAM_ID,
      "TokenkegQfeZyiNwAJbNbGKPFXCWuBvf9Ss623VQ5DA",
      // Token Program
      "ATokenGPvbdGVxr1b2hvZbsiqW5xWH25efTNsLJA8knL",
      // Associated Token Program
      "11111111111111111111111111111111",
      // System Program
      "SysvarRent111111111111111111111111111111111",
      // Rent Sysvar
      "metaqbxxUerdq28cj1RbAWkYQm3ybzjb6a8bt518x1s",
      // Metaplex Metadata
      "Ce6TQqeHC9p8KetsN6JsjHK7UTZk7nasjjnr7XxXp9F1"
      // Pumpfun Fee Account
    ]);
    for (let i = 1; i < Math.min(10, accountKeys.length); i++) {
      const key = typeof accountKeys[i] === "string" ? accountKeys[i] : (_f = accountKeys[i]) == null ? void 0 : _f.pubkey;
      if (key && !excludedAddresses.has(key) && key.length >= 32 && key.length <= 44) {
        return key;
      }
    }
  } catch {
  }
  return null;
}
function calculateScore(cluster, rugAnalysis, dexAnalysis) {
  const tokenCount = cluster.tokensCreated.length;
  const rugRatio = rugAnalysis.rugRatio;
  const dexSuccessRate = dexAnalysis.successRate;
  const bundleWalletCount = cluster.bundleWallets.length;
  const isInsufficientData = tokenCount < 3;
  let tokenCountScore = 50;
  if (tokenCount === 0) tokenCountScore = 30;
  else if (tokenCount <= 2) tokenCountScore = 60;
  else if (tokenCount <= 5) tokenCountScore = 85;
  else if (tokenCount <= 10) tokenCountScore = 75;
  else if (tokenCount <= 15) tokenCountScore = 55;
  else if (tokenCount <= 30) tokenCountScore = 35;
  else tokenCountScore = 15;
  let rugRatioScore;
  if (isInsufficientData) {
    rugRatioScore = Math.round((1 - rugRatio) * 100 * 0.7);
  } else {
    rugRatioScore = Math.round((1 - rugRatio) * 100);
  }
  let dexSuccessScore;
  if (isInsufficientData) {
    dexSuccessScore = Math.round(dexSuccessRate * 100 * 0.7);
  } else {
    dexSuccessScore = Math.round(dexSuccessRate * 100);
  }
  let bundleScore = 100;
  if (bundleWalletCount >= 5) bundleScore = 20;
  else if (bundleWalletCount >= 3) bundleScore = 40;
  else if (bundleWalletCount >= 1) bundleScore = 70;
  let overall = Math.round(
    tokenCountScore * 0.15 + rugRatioScore * 0.45 + dexSuccessScore * 0.3 + bundleScore * 0.1
  );
  if (isInsufficientData && overall > 75) {
    overall = 75;
  }
  let grade;
  let label;
  if (overall >= 80) grade = "A";
  else if (overall >= 60) grade = "B";
  else if (overall >= 40) grade = "C";
  else if (overall >= 20) grade = "D";
  else grade = "F";
  if (tokenCount < 3) {
    label = "新号/数据不足";
  } else if (tokenCount >= 10 && rugRatio >= 0.5) {
    label = "地毯工厂";
    if (grade !== "F") grade = "D";
  } else if (tokenCount >= 15) {
    label = "高频农夫";
  } else if (dexSuccessRate >= 0.3 && rugRatio < 0.3 && tokenCount >= 3) {
    label = "认真开发";
  } else {
    label = "中等活跃";
  }
  return {
    overall,
    grade,
    label,
    breakdown: {
      tokenCount: { value: tokenCount, score: tokenCountScore, weight: 0.15 },
      rugRatio: { value: rugRatio, score: rugRatioScore, weight: 0.45 },
      dexSuccessRate: { value: dexSuccessRate, score: dexSuccessScore, weight: 0.3 },
      bundleWalletCount: { value: bundleWalletCount, score: bundleScore, weight: 0.1 }
    }
  };
}
function createErrorResult(devAddress, error) {
  return {
    devAddress,
    cluster: { primaryDev: devAddress, fundingSources: [], bundleWallets: [], tokensCreated: [] },
    rugAnalysis: { suspectedRugCount: 0, rugRatio: 0, rugTokens: [] },
    dexAnalysis: { listedCount: 0, successRate: 0, totalTokens: 0, tokens: [] },
    score: {
      overall: 50,
      grade: "C",
      label: "新号/数据不足",
      breakdown: {
        tokenCount: { value: 0, score: 50, weight: 0.15 },
        rugRatio: { value: 0, score: 100, weight: 0.45 },
        dexSuccessRate: { value: 0, score: 0, weight: 0.4 },
        bundleWalletCount: { value: 0, score: 100, weight: 0 }
      }
    },
    analyzedAt: Date.now(),
    error: error instanceof Error ? error.message : "分析失败"
  };
}
function broadcastToSidepanel(message) {
  chrome.runtime.sendMessage(message).catch(() => {
  });
}
console.log("Dev Radar Background Service Worker 已启动");
