function isValidSolanaAddress(address) {
  if (!address || address.length < 32 || address.length > 44) return false;
  const base58Regex = /^[1-9A-HJ-NP-Za-km-z]{32,44}$/;
  return base58Regex.test(address);
}
function extractTokenFromUrl(url) {
  const patterns = [
    /\/sol\/token\/([1-9A-HJ-NP-Za-km-z]{32,44})/,
    /\/token\/([1-9A-HJ-NP-Za-km-z]{32,44})/,
    /[?&]token=([1-9A-HJ-NP-Za-km-z]{32,44})/,
    /[?&]mint=([1-9A-HJ-NP-Za-km-z]{32,44})/
  ];
  for (const pattern of patterns) {
    const match = url.match(pattern);
    if (match && isValidSolanaAddress(match[1])) {
      return match[1];
    }
  }
  return null;
}
function debounce(fn, delay) {
  let timeoutId;
  return function(...args) {
    clearTimeout(timeoutId);
    timeoutId = setTimeout(() => fn(...args), delay);
  };
}
let lastDevAddress = null;
let lastTokenMint = null;
function init() {
  console.log("[Dev Radar] Content Script 已加载");
  console.log("[Dev Radar] 当前 URL:", location.href);
  setTimeout(() => {
    detectDevInfo();
  }, 1e3);
  observeDOMChanges();
  observeUrlChanges();
}
function observeDOMChanges() {
  const debouncedDetect = debounce(() => {
    detectDevInfo();
  }, 800);
  const observer = new MutationObserver(() => {
    debouncedDetect();
  });
  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}
function observeUrlChanges() {
  let lastUrl = location.href;
  const checkUrl = () => {
    if (location.href !== lastUrl) {
      console.log("[Dev Radar] URL 变化:", lastUrl, "->", location.href);
      lastUrl = location.href;
      lastDevAddress = null;
      lastTokenMint = null;
      setTimeout(() => {
        detectDevInfo();
      }, 1500);
    }
  };
  setInterval(checkUrl, 500);
  window.addEventListener("popstate", () => {
    lastDevAddress = null;
    lastTokenMint = null;
    setTimeout(() => detectDevInfo(), 1e3);
  });
}
function detectDevInfo() {
  console.log("[Dev Radar] ========== 开始检测 ==========");
  console.log("[Dev Radar] 当前 URL:", location.href);
  const tokenMint = extractTokenFromUrl(location.href);
  console.log("[Dev Radar] 提取的 Token Mint:", tokenMint);
  if (!tokenMint) {
    console.log("[Dev Radar] 未从 URL 提取到 Token 地址");
    return;
  }
  let devAddress = findDevAddress();
  console.log("[Dev Radar] 找到的 Dev 地址:", devAddress);
  if (!devAddress) {
    console.log("[Dev Radar] 未找到 Dev 地址，使用 Token 地址作为分析对象");
    devAddress = tokenMint;
  }
  if (devAddress === lastDevAddress && tokenMint === lastTokenMint) {
    console.log("[Dev Radar] 地址未变化，跳过");
    return;
  }
  lastDevAddress = devAddress;
  lastTokenMint = tokenMint;
  const tokenInfo = findTokenInfo();
  const devInfo = {
    devAddress,
    tokenMint,
    tokenName: tokenInfo.tokenName,
    tokenSymbol: tokenInfo.tokenSymbol
  };
  console.log("[Dev Radar] ========== 发送消息 ==========");
  console.log("[Dev Radar] devInfo:", JSON.stringify(devInfo, null, 2));
  chrome.runtime.sendMessage({
    type: "DEV_INFO_DETECTED",
    payload: devInfo
  }).then(() => {
    console.log("[Dev Radar] 消息发送成功");
  }).catch((err) => {
    console.error("[Dev Radar] 发送消息失败:", err);
  });
}
function findDevAddress() {
  var _a, _b, _c, _d, _e, _f;
  const devSelectors = [
    // 查找包含 dev/creator 文本的元素附近的地址
    'a[href*="solscan.io/account"]',
    'a[href*="explorer.solana.com/address"]',
    'a[href*="/sol/address/"]'
  ];
  for (const selector of devSelectors) {
    const links = document.querySelectorAll(selector);
    for (const link of links) {
      const href = link.getAttribute("href") || "";
      const container = link.closest("div, span, td, li");
      const containerText = ((_a = container == null ? void 0 : container.textContent) == null ? void 0 : _a.toLowerCase()) || "";
      if (containerText.includes("dev") || containerText.includes("creator") || containerText.includes("开发者") || containerText.includes("创建者")) {
        const addressMatch = href.match(/([1-9A-HJ-NP-Za-km-z]{32,44})/);
        if (addressMatch && isValidSolanaAddress(addressMatch[1])) {
          console.log("[Dev Radar] 从 Dev 相关链接找到地址:", addressMatch[1]);
          return addressMatch[1];
        }
      }
    }
  }
  const allElements = document.querySelectorAll("*");
  for (const el of allElements) {
    const text = ((_b = el.textContent) == null ? void 0 : _b.trim()) || "";
    if (text === "Dev" || text === "DEV" || text === "开发者" || text === "Creator") {
      const parent = el.parentElement;
      if (parent) {
        const address = extractAddressFromContainer(parent);
        if (address) {
          console.log("[Dev Radar] 从 Dev 标签附近找到地址:", address);
          return address;
        }
        const grandParent = parent.parentElement;
        if (grandParent) {
          const address2 = extractAddressFromContainer(grandParent);
          if (address2) {
            console.log("[Dev Radar] 从 Dev 标签祖先找到地址:", address2);
            return address2;
          }
        }
      }
    }
  }
  const explorerLinks = document.querySelectorAll(
    'a[href*="solscan.io"], a[href*="explorer.solana.com"], a[href*="/sol/address/"]'
  );
  for (const link of explorerLinks) {
    const href = link.getAttribute("href") || "";
    const match = href.match(/(?:account|address)\/([1-9A-HJ-NP-Za-km-z]{32,44})/);
    if (match && isValidSolanaAddress(match[1])) {
      const linkText = ((_c = link.textContent) == null ? void 0 : _c.toLowerCase()) || "";
      const linkTitle = ((_d = link.getAttribute("title")) == null ? void 0 : _d.toLowerCase()) || "";
      const nearbyText = ((_f = (_e = link.closest("div")) == null ? void 0 : _e.textContent) == null ? void 0 : _f.toLowerCase()) || "";
      if (linkText.includes("dev") || linkTitle.includes("dev") || nearbyText.includes("dev") || nearbyText.includes("开发者")) {
        console.log("[Dev Radar] 从 explorer 链接找到 Dev 地址:", match[1]);
        return match[1];
      }
    }
  }
  const dataElements = document.querySelectorAll("[data-address], [data-wallet], [data-creator]");
  for (const el of dataElements) {
    const address = el.getAttribute("data-address") || el.getAttribute("data-wallet") || el.getAttribute("data-creator");
    if (address && isValidSolanaAddress(address)) {
      console.log("[Dev Radar] 从 data 属性找到地址:", address);
      return address;
    }
  }
  const urlParams = new URLSearchParams(location.search);
  const devParam = urlParams.get("dev") || urlParams.get("creator") || urlParams.get("developer");
  if (devParam && isValidSolanaAddress(devParam)) {
    console.log("[Dev Radar] 从 URL 参数找到 Dev 地址:", devParam);
    return devParam;
  }
  return null;
}
function extractAddressFromContainer(container) {
  const links = container.querySelectorAll("a");
  for (const link of links) {
    const href = link.getAttribute("href") || "";
    const match = href.match(/([1-9A-HJ-NP-Za-km-z]{32,44})/);
    if (match && isValidSolanaAddress(match[1])) {
      return match[1];
    }
  }
  const text = container.textContent || "";
  const addressMatch = text.match(/[1-9A-HJ-NP-Za-km-z]{32,44}/);
  if (addressMatch && isValidSolanaAddress(addressMatch[0])) {
    return addressMatch[0];
  }
  const dataAddress = container.getAttribute("data-address") || container.getAttribute("title");
  if (dataAddress && isValidSolanaAddress(dataAddress)) {
    return dataAddress;
  }
  return null;
}
function findTokenInfo() {
  var _a, _b;
  const title = document.title;
  const titleMatch = title.match(/^([^|–-]+)/);
  const h1 = document.querySelector("h1");
  const h1Text = ((_a = h1 == null ? void 0 : h1.textContent) == null ? void 0 : _a.trim()) || "";
  const symbolMatch = (h1Text + " " + title).match(/\$([A-Za-z0-9]+)/);
  return {
    tokenName: h1Text || ((_b = titleMatch == null ? void 0 : titleMatch[1]) == null ? void 0 : _b.trim()),
    tokenSymbol: symbolMatch == null ? void 0 : symbolMatch[1]
  };
}
chrome.runtime.onMessage.addListener((message, _sender, sendResponse) => {
  if (message.type === "REQUEST_DETECTION") {
    console.log("[Dev Radar] 收到检测请求");
    lastDevAddress = null;
    lastTokenMint = null;
    detectDevInfo();
    sendResponse({ success: true });
  }
  return true;
});
if (document.readyState === "loading") {
  document.addEventListener("DOMContentLoaded", init);
} else {
  init();
}
